// @ts-check
const { defineConfig, devices } = require('@playwright/test');

/**
 * PWB E2E Testing Configuration
 * @see https://playwright.dev/docs/test-configuration
 *
 * Before running tests, ensure the e2e database is set up:
 *   RAILS_ENV=e2e bin/rails playwright:reset
 *
 * Start the e2e server:
 *   RAILS_ENV=e2e bin/rails playwright:server
 */
module.exports = defineConfig({
  testDir: './tests/e2e',

  // Global setup - verifies e2e database is properly initialized
  globalSetup: './tests/e2e/global-setup.js',
  
  // Run tests in files in parallel
  fullyParallel: true,
  
  // Fail the build on CI if you accidentally left test.only in the source code
  forbidOnly: !!process.env.CI,
  
  // Retry on CI only
  retries: process.env.CI ? 2 : 0,
  
  // Opt out of parallel tests on CI
  workers: process.env.CI ? 1 : undefined,
  
  // Reporter to use
  reporter: [
    ['html', { outputFolder: 'playwright-report' }],
    ['list']
  ],
  
  // Shared settings for all the projects below
  use: {
    // Base URL for tests (e2e environment runs on 3001)
    baseURL: 'http://tenant-a.e2e.localhost:3001',
    
    // Collect trace when retrying the failed test
    trace: 'on-first-retry',
    
    // Screenshot on failure
    screenshot: 'only-on-failure',
    
    // Video on retry
    video: 'on-first-retry',
  },

  // Configure projects for major browsers
  projects: [
    {
      name: 'chromium',
      use: { ...devices['Desktop Chrome'] },
      // Exclude admin tests - they run in a separate serial project
      testIgnore: '**/admin/**',
    },
    {
      // Admin tests modify shared state and must run serially
      name: 'chromium-admin',
      use: { ...devices['Desktop Chrome'] },
      testMatch: '**/admin/**',
      fullyParallel: false,
    },
    // Uncomment for additional browser testing
    // {
    //   name: 'firefox',
    //   use: { ...devices['Desktop Firefox'] },
    // },
    // {
    //   name: 'webkit',
    //   use: { ...devices['Desktop Safari'] },
    // },
  ],

  // Note: The Rails server should be started manually using:
  // RAILS_ENV=e2e bin/rails playwright:server
  // This is preferred over webServer config for Rails apps
  // webServer: {
  //   command: 'RAILS_ENV=e2e bin/rails server -p 3001',
  //   url: 'http://tenant-a.e2e.localhost:3001',
  //   reuseExistingServer: !process.env.CI,
  //   timeout: 120 * 1000,
  // },
});
