# frozen_string_literal: true

module Pwb
  # FieldKey records translation keys used by fields like propType, propState, features.
  #
  # Note: This model is NOT tenant-scoped. Use PwbTenant::FieldKey for
  # tenant-scoped queries in web requests. This version is useful for
  # console work and cross-tenant operations.
  #
  # The table uses `pwb_website_id` as the foreign key column (not `website_id`).
# == Schema Information
#
# Table name: pwb_field_keys
#
#  id                  :integer          not null
#  global_key          :string           primary key
#  props_count         :integer          default(0), not null
#  show_in_search_form :boolean          default(TRUE)
#  sort_order          :integer          default(0)
#  tag                 :string
#  translations        :jsonb            not null
#  visible             :boolean          default(TRUE)
#  created_at          :datetime         not null
#  updated_at          :datetime         not null
#  pwb_website_id      :bigint
#
# Indexes
#
#  index_field_keys_on_website_and_tag     (pwb_website_id,tag)
#  index_field_keys_unique_per_website     (pwb_website_id,global_key) UNIQUE
#  index_pwb_field_keys_on_pwb_website_id  (pwb_website_id)
#
# Foreign Keys
#
#  fk_rails_...  (pwb_website_id => pwb_websites.id)
#
  #
  class FieldKey < ApplicationRecord
    extend Mobility

    self.table_name = 'pwb_field_keys'
    self.primary_key = :global_key

    # Translatable attribute - stored in JSONB 'translations' column
    # Provides: label, label_en, label_es, etc. with automatic fallbacks
    translates :label

    belongs_to :website, class_name: 'Pwb::Website', foreign_key: 'pwb_website_id', optional: true

    # Scopes
    scope :visible, -> { where(visible: true) }
    scope :by_tag, ->(tag) { where(tag: tag) }
    scope :ordered, -> { order(:sort_order, :created_at) }

    # Validations
    # Note: uniqueness is scoped by pwb_website_id (the actual column name)
    validates :global_key, presence: true, uniqueness: { scope: :pwb_website_id }
    validates :tag, presence: true

    # RealtyAsset associations
    has_many :realty_assets_with_state, class_name: 'Pwb::RealtyAsset', foreign_key: 'prop_state_key', primary_key: :global_key
    has_many :realty_assets_with_type, class_name: 'Pwb::RealtyAsset', foreign_key: 'prop_type_key', primary_key: :global_key

    has_many :features, class_name: 'Pwb::Feature', foreign_key: 'feature_key', primary_key: :global_key

    # Get values to populate dropdowns in search forms.
    # Returns an array of OpenStruct objects with :value and :label.
    #
    # Note: This method does NOT automatically scope by tenant. When called from
    # a web context, use PwbTenant::FieldKey.get_options_by_tag instead, or
    # ensure you're calling within an ActsAsTenant.with_tenant block.
    #
    # @param tag [String] The tag to filter by (e.g., 'property-types')
    # @return [Array<OpenStruct>] Options sorted by sort_order, then created_at
    #
    def self.get_options_by_tag(tag)
      where(tag: tag)
        .visible
        .ordered
        .map do |field_key|
          OpenStruct.new(
            value: field_key.global_key,
            label: field_key.display_label,
            sort_order: field_key.sort_order
          )
        end
    end

    # Returns the display label for this field key.
    # Uses Mobility's translated label with fallback to global_key.
    #
    # @return [String] The localized label or the global_key as fallback
    #
    def display_label
      label.presence || global_key
    end
  end
end
