# frozen_string_literal: true

module SiteAdmin
  module Website
    class SettingsController < ::SiteAdminController
      before_action :set_website
      before_action :set_tab

      VALID_TABS = %w[general appearance navigation home notifications seo social].freeze

      def show
        # Always load website locales for multilingual editing
        @website_locales = build_website_locales

        case @tab
        when 'navigation'
          @top_nav_links = @website.links.ordered_top_nav
          @footer_links = @website.links.ordered_footer
        when 'home'
          @home_page = @website.pages.find_by(slug: 'home')
          @carousel_contents = @website.contents.where(tag: 'carousel')
        when 'appearance'
          # Only show themes that are accessible to this website
          @themes = @website.accessible_themes
          @style_variables = @website.style_variables
        when 'seo'
          @social_media = @website.social_media || {}
        when 'social'
          @social_links = @website.social_media_links_for_admin
        end
      end

      def update
        case @tab
        when 'general'
          update_general_settings
        when 'appearance'
          update_appearance_settings
        when 'home'
          update_home_settings
        when 'notifications'
          update_notification_settings
        when 'seo'
          update_seo_settings
        when 'social'
          update_social_settings
        else
          redirect_to site_admin_website_settings_path, alert: 'Invalid tab'
          return
        end
      end

      def test_notifications
        result = NtfyService.test_configuration(@website)
        if result[:success]
          redirect_to site_admin_website_settings_tab_path('notifications'), notice: result[:message]
        else
          redirect_to site_admin_website_settings_tab_path('notifications'), alert: result[:message]
        end
      end

      def update_links
        if params[:links].present?
          update_navigation_links
          redirect_to site_admin_website_settings_tab_path('navigation'), notice: 'Navigation links updated successfully'
        else
          redirect_to site_admin_website_settings_tab_path('navigation'), alert: 'No links data provided'
        end
      end

      private

      def set_website
        @website = current_website
      end

      def set_tab
        @tab = params[:tab].presence || 'general'
        unless VALID_TABS.include?(@tab)
          redirect_to site_admin_website_settings_path, alert: 'Invalid tab'
        end
      end

      def update_general_settings
        # Filter out blank values from supported_locales (the hidden field submits "")
        filtered_params = general_settings_params.to_h
        if filtered_params[:supported_locales].is_a?(Array)
          filtered_params[:supported_locales] = filtered_params[:supported_locales].reject(&:blank?)
        end

        if @website.update(filtered_params)
          redirect_to site_admin_website_settings_tab_path('general'), notice: 'General settings updated successfully'
        else
          flash.now[:alert] = 'Failed to update general settings'
          render :show, status: :unprocessable_entity
        end
      end

      def update_appearance_settings
        # Handle style_variables separately since it's a nested structure
        if params[:website][:style_variables].present?
          @website.style_variables = params[:website][:style_variables].to_unsafe_h
        end

        if @website.update(appearance_settings_params)
          redirect_to site_admin_website_settings_tab_path('appearance'), notice: 'Appearance settings updated successfully'
        else
          @themes = @website.accessible_themes
          @style_variables = @website.style_variables
          flash.now[:alert] = 'Failed to update appearance settings'
          render :show, status: :unprocessable_entity
        end
      end

      def update_home_settings
        # Handle page title updates
        if params[:page].present?
          @home_page = @website.pages.find_by(slug: 'home')
          if @home_page&.update(home_page_params)
            redirect_to site_admin_website_settings_tab_path('home'), notice: 'Home page title updated successfully'
            return
          else
            @carousel_contents = @website.contents.where(tag: 'carousel')
            flash.now[:alert] = 'Failed to update home page title'
            render :show, status: :unprocessable_entity
            return
          end
        end

        # Handle website display options (landing_hide_* flags)
        if params[:website].present?
          if @website.update(home_display_params)
            redirect_to site_admin_website_settings_tab_path('home'), notice: 'Display options updated successfully'
          else
            @home_page = @website.pages.find_by(slug: 'home')
            @carousel_contents = @website.contents.where(tag: 'carousel')
            flash.now[:alert] = 'Failed to update display options'
            render :show, status: :unprocessable_entity
          end
        else
          redirect_to site_admin_website_settings_tab_path('home')
        end
      end

      def update_notification_settings
        # Don't clear the access token if the placeholder is submitted
        filtered_params = notification_settings_params.to_h
        if filtered_params[:ntfy_access_token] == '••••••••••••' || filtered_params[:ntfy_access_token].blank?
          filtered_params.delete(:ntfy_access_token)
        end

        if @website.update(filtered_params)
          redirect_to site_admin_website_settings_tab_path('notifications'), notice: 'Notification settings updated successfully'
        else
          flash.now[:alert] = 'Failed to update notification settings'
          render :show, status: :unprocessable_entity
        end
      end

      def update_seo_settings
        # Merge social_media settings
        if params[:social_media].present?
          current_social = @website.social_media || {}
          @website.social_media = current_social.merge(params[:social_media].to_unsafe_h)
        end

        if @website.update(seo_settings_params)
          redirect_to site_admin_website_settings_tab_path('seo'), notice: 'SEO settings updated successfully'
        else
          @social_media = @website.social_media || {}
          flash.now[:alert] = 'Failed to update SEO settings'
          render :show, status: :unprocessable_entity
        end
      end

      def update_social_settings
        if params[:social_links].present?
          params[:social_links].each do |platform, url|
            @website.update_social_media_link(platform, url)
          end
          redirect_to site_admin_website_settings_tab_path('social'), notice: 'Social media links updated successfully'
        else
          redirect_to site_admin_website_settings_tab_path('social'), alert: 'No social media data provided'
        end
      end

      def update_navigation_links
        params[:links].each do |link_params|
          link = @website.links.find_by(id: link_params[:id])
          next unless link

          # Build update hash
          update_attrs = {
            link_path: link_params[:link_path],
            visible: link_params[:visible] == 'true' || link_params[:visible] == true,
            sort_order: link_params[:sort_order]
          }

          # Handle multilingual titles if provided
          if link_params[:titles].present?
            link_params[:titles].each do |locale, title|
              # Use Mobility's locale-specific setter
              Mobility.with_locale(locale.to_sym) do
                link.link_title = title
              end
            end
            link.save
          else
            # Fallback to single title for backwards compatibility
            update_attrs[:link_title] = link_params[:link_title] if link_params[:link_title].present?
            link.update(update_attrs)
          end
        end
      end

      def general_settings_params
        # Form submits as pwb_website (from model class name), but we also check :website for flexibility
        param_key = params.key?(:pwb_website) ? :pwb_website : :website
        params.require(param_key).permit(
          :company_display_name,
          :default_client_locale,
          :default_currency,
          :default_area_unit,
          :analytics_id,
          :analytics_id_type,
          :external_image_mode,
          supported_locales: [],
          available_currencies: []
        )
      end

      def appearance_settings_params
        params.require(:website).permit(
          :theme_name,
          :selected_palette,
          :dark_mode_setting,
          :raw_css
        )
      end

      def home_page_params
        params.require(:page).permit(:page_title)
      end

      def home_display_params
        params.require(:website).permit(
          :landing_hide_for_rent,
          :landing_hide_for_sale,
          :landing_hide_search_bar
        )
      end

      def notification_settings_params
        param_key = params.key?(:pwb_website) ? :pwb_website : :website
        params.require(param_key).permit(
          :ntfy_enabled,
          :ntfy_server_url,
          :ntfy_topic_prefix,
          :ntfy_access_token,
          :ntfy_notify_inquiries,
          :ntfy_notify_listings,
          :ntfy_notify_users,
          :ntfy_notify_security
        )
      end

      def seo_settings_params
        param_key = params.key?(:pwb_website) ? :pwb_website : :website
        params.require(param_key).permit(
          :default_seo_title,
          :default_meta_description,
          :favicon_url,
          :main_logo_url
        )
      end

      # Build locale details for the website's supported locales
      # Uses Pwb::Config for centralized locale configuration
      def build_website_locales
        supported = @website.supported_locales || ['en']
        Pwb::Config.build_locale_details(supported)
      end
    end
  end
end
